'use client';

import React, { useState, useEffect } from 'react';
import { ArrowLeft, Plus, Search, User, Shield, Eye, Edit, Trash2, UserPlus, Settings } from 'lucide-react';
import Link from 'next/link';
import { useParams } from 'next/navigation';
import type { Customer, CorporateUser, UserPermission } from '@/types';
import PermissionGuard from '@/components/guards/PermissionGuard';

export default function CorporateUsersPage() {
  return (
    <PermissionGuard requiredPermissions={['customers_view', 'customers_view_own']}>
      <CorporateUsersPageContent />
    </PermissionGuard>
  );
}

function CorporateUsersPageContent() {
  const params = useParams();
  const customerId = params.id as string;
  
  const [customer, setCustomer] = useState<Customer | null>(null);
  const [users, setUsers] = useState<CorporateUser[]>([]);
  const [loading, setLoading] = useState(true);
  const [searchTerm, setSearchTerm] = useState('');
  const [showAddUser, setShowAddUser] = useState(false);

  useEffect(() => {
    fetchCustomerAndUsers();
  }, [customerId]);

  const fetchCustomerAndUsers = async () => {
    try {
      setLoading(true);
      
      // Mock data - gerçek API ile değiştirilecek
      const mockCustomer: Customer = {
        id: customerId,
        type: 'corporate',
        companyName: 'ABC Teknoloji A.Ş.',
        taxNumber: '1234567890',
        taxOffice: 'Beşiktaş',
        email: 'info@abcteknoloji.com',
        phone: '+90 212 555 0123',
        address: 'Levent Mahallesi, İş Kuleleri',
        city: 'İstanbul',
        contactPerson: 'Fatma Kaya',
        department: 'İnsan Kaynakları',
        isVip: false,
        isActive: true,
        createdAt: new Date('2024-03-10'),
        updatedAt: new Date('2024-12-18')
      };

      const mockUsers: CorporateUser[] = [
        {
          id: '1',
          customerId: customerId,
          firstName: 'Fatma',
          lastName: 'Kaya',
          email: 'fatma.kaya@abcteknoloji.com',
          phone: '+90 555 123 4567',
          department: 'İnsan Kaynakları',
          position: 'İK Müdürü',
          permissions: ['create_ticket_request', 'create_accommodation_request', 'create_visa_request', 'create_transfer_request', 'view_all_requests', 'manage_users'],
          isActive: true,
          lastLoginAt: new Date('2024-12-20'),
          createdAt: new Date('2024-03-15'),
          updatedAt: new Date('2024-12-20')
        },
        {
          id: '2',
          customerId: customerId,
          firstName: 'Mehmet',
          lastName: 'Özkan',
          email: 'mehmet.ozkan@abcteknoloji.com',
          phone: '+90 555 987 6543',
          department: 'Satış',
          position: 'Satış Temsilcisi',
          permissions: ['create_ticket_request', 'create_accommodation_request', 'view_own_requests'],
          isActive: true,
          lastLoginAt: new Date('2024-12-19'),
          createdAt: new Date('2024-04-20'),
          updatedAt: new Date('2024-12-19')
        },
        {
          id: '3',
          customerId: customerId,
          firstName: 'Ayşe',
          lastName: 'Demir',
          email: 'ayse.demir@abcteknoloji.com',
          department: 'Pazarlama',
          position: 'Pazarlama Uzmanı',
          permissions: ['create_ticket_request', 'view_own_requests'],
          isActive: false,
          createdAt: new Date('2024-06-10'),
          updatedAt: new Date('2024-11-15')
        }
      ];

      setCustomer(mockCustomer);
      setUsers(mockUsers);
    } catch (error) {
      console.error('Error fetching data:', error);
    } finally {
      setLoading(false);
    }
  };

  const getPermissionText = (permission: UserPermission): string => {
    const permissionMap = {
      'create_ticket_request': 'Bilet Talebi Oluştur',
      'create_accommodation_request': 'Konaklama Talebi Oluştur',
      'create_visa_request': 'Vize Talebi Oluştur',
      'create_transfer_request': 'Transfer Talebi Oluştur',
      'view_all_requests': 'Tüm Talepleri Görüntüle',
      'view_own_requests': 'Kendi Taleplerini Görüntüle',
      'manage_users': 'Kullanıcı Yönetimi'
    };
    return permissionMap[permission] || permission;
  };

  const filteredUsers = users.filter(user => {
    const searchText = `${user.firstName} ${user.lastName} ${user.email} ${user.department || ''}`.toLowerCase();
    return searchText.includes(searchTerm.toLowerCase());
  });

  const stats = {
    total: users.length,
    active: users.filter(u => u.isActive).length,
    inactive: users.filter(u => !u.isActive).length,
    admins: users.filter(u => u.permissions.includes('manage_users')).length
  };

  if (loading) {
    return (
      <div className="p-6">
        <div className="animate-pulse space-y-4">
          <div className="h-8 bg-gray-200 rounded w-1/4"></div>
          <div className="h-64 bg-gray-200 rounded"></div>
        </div>
      </div>
    );
  }

  if (!customer || customer.type !== 'corporate') {
    return (
      <div className="p-6">
        <div className="text-center py-12">
          <h3 className="text-lg font-medium text-gray-900">Müşteri bulunamadı</h3>
          <p className="text-gray-500">Bu sayfa sadece kurumsal müşteriler için geçerlidir.</p>
          <Link href="/customers" className="mt-4 text-blue-600 hover:text-blue-800">
            Müşteri listesine dön
          </Link>
        </div>
      </div>
    );
  }

  return (
    <div className="p-6">
      {/* Header */}
      <div className="flex items-center justify-between mb-6">
        <div className="flex items-center space-x-4">
          <Link 
            href="/customers"
            className="p-2 hover:bg-gray-100 rounded-lg transition-colors"
          >
            <ArrowLeft className="h-5 w-5 text-gray-600" />
          </Link>
          <div>
            <h1 className="text-2xl font-bold text-gray-900 flex items-center">
              <User className="mr-3 h-8 w-8 text-purple-600" />
              {customer.companyName} - Kullanıcılar
            </h1>
            <p className="text-gray-600 mt-1">
              Kurumsal müşteri kullanıcılarını yönetin
            </p>
          </div>
        </div>
        <button 
          onClick={() => setShowAddUser(true)}
          className="bg-purple-600 text-white px-4 py-2 rounded-lg hover:bg-purple-700 flex items-center"
        >
          <UserPlus className="mr-2 h-4 w-4" />
          Yeni Kullanıcı
        </button>
      </div>

      {/* Company Info */}
      <div className="bg-white rounded-lg border border-gray-200 p-6 mb-6">
        <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
          <div>
            <h3 className="text-sm font-medium text-gray-500">Şirket Bilgileri</h3>
            <p className="text-lg font-semibold text-gray-900">{customer.companyName}</p>
            <p className="text-sm text-gray-600">VN: {customer.taxNumber}</p>
          </div>
          <div>
            <h3 className="text-sm font-medium text-gray-500">İletişim Kişisi</h3>
            <p className="text-lg font-semibold text-gray-900">{customer.contactPerson}</p>
            <p className="text-sm text-gray-600">{customer.department}</p>
          </div>
          <div>
            <h3 className="text-sm font-medium text-gray-500">İletişim</h3>
            <p className="text-sm text-gray-900">{customer.email}</p>
            <p className="text-sm text-gray-600">{customer.phone}</p>
          </div>
        </div>
      </div>

      {/* Stats Cards */}
      <div className="grid grid-cols-1 md:grid-cols-4 gap-4 mb-6">
        <div className="bg-white p-4 rounded-lg shadow-sm border border-gray-200">
          <div className="flex items-center">
            <User className="h-8 w-8 text-blue-600" />
            <div className="ml-3">
              <p className="text-sm font-medium text-gray-500">Toplam Kullanıcı</p>
              <p className="text-2xl font-semibold text-gray-900">{stats.total}</p>
            </div>
          </div>
        </div>
        <div className="bg-white p-4 rounded-lg shadow-sm border border-gray-200">
          <div className="flex items-center">
            <div className="h-8 w-8 bg-green-100 rounded-full flex items-center justify-center">
              <div className="h-3 w-3 bg-green-600 rounded-full"></div>
            </div>
            <div className="ml-3">
              <p className="text-sm font-medium text-gray-500">Aktif</p>
              <p className="text-2xl font-semibold text-gray-900">{stats.active}</p>
            </div>
          </div>
        </div>
        <div className="bg-white p-4 rounded-lg shadow-sm border border-gray-200">
          <div className="flex items-center">
            <div className="h-8 w-8 bg-red-100 rounded-full flex items-center justify-center">
              <div className="h-3 w-3 bg-red-600 rounded-full"></div>
            </div>
            <div className="ml-3">
              <p className="text-sm font-medium text-gray-500">Pasif</p>
              <p className="text-2xl font-semibold text-gray-900">{stats.inactive}</p>
            </div>
          </div>
        </div>
        <div className="bg-white p-4 rounded-lg shadow-sm border border-gray-200">
          <div className="flex items-center">
            <Shield className="h-8 w-8 text-purple-600" />
            <div className="ml-3">
              <p className="text-sm font-medium text-gray-500">Yönetici</p>
              <p className="text-2xl font-semibold text-gray-900">{stats.admins}</p>
            </div>
          </div>
        </div>
      </div>

      {/* Search */}
      <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-4 mb-6">
        <div className="relative">
          <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-400 h-4 w-4" />
          <input
            type="text"
            placeholder="Kullanıcı adı, email veya departman ara..."
            className="w-full pl-10 pr-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-purple-500 focus:border-transparent"
            value={searchTerm}
            onChange={(e) => setSearchTerm(e.target.value)}
          />
        </div>
      </div>

      {/* Users List */}
      <div className="bg-white rounded-lg shadow-sm border border-gray-200">
        <div className="overflow-x-auto">
          <table className="min-w-full divide-y divide-gray-200">
            <thead className="bg-gray-50">
              <tr>
                <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                  Kullanıcı
                </th>
                <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                  Departman/Pozisyon
                </th>
                <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                  Yetkiler
                </th>
                <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                  Son Giriş
                </th>
                <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                  Durum
                </th>
                <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                  İşlemler
                </th>
              </tr>
            </thead>
            <tbody className="bg-white divide-y divide-gray-200">
              {filteredUsers.map((user) => (
                <tr key={user.id} className="hover:bg-gray-50">
                  <td className="px-6 py-4 whitespace-nowrap">
                    <div className="flex items-center">
                      <div className="flex-shrink-0 h-10 w-10">
                        <div className="h-10 w-10 rounded-full bg-purple-100 flex items-center justify-center">
                          <span className="text-sm font-medium text-purple-600">
                            {user.firstName[0]}{user.lastName[0]}
                          </span>
                        </div>
                      </div>
                      <div className="ml-4">
                        <div className="text-sm font-medium text-gray-900 flex items-center">
                          {user.firstName} {user.lastName}
                          {user.permissions.includes('manage_users') && (
                            <Shield className="ml-2 h-4 w-4 text-purple-500" />
                          )}
                        </div>
                        <div className="text-sm text-gray-500">{user.email}</div>
                        {user.phone && (
                          <div className="text-xs text-gray-400">{user.phone}</div>
                        )}
                      </div>
                    </div>
                  </td>
                  <td className="px-6 py-4 whitespace-nowrap">
                    <div className="text-sm text-gray-900">{user.department}</div>
                    {user.position && (
                      <div className="text-sm text-gray-500">{user.position}</div>
                    )}
                  </td>
                  <td className="px-6 py-4">
                    <div className="flex flex-wrap gap-1">
                      {user.permissions.slice(0, 2).map((permission) => (
                        <span
                          key={permission}
                          className="inline-flex px-2 py-1 text-xs font-medium bg-blue-100 text-blue-800 rounded-full"
                        >
                          {getPermissionText(permission)}
                        </span>
                      ))}
                      {user.permissions.length > 2 && (
                        <span className="inline-flex px-2 py-1 text-xs font-medium bg-gray-100 text-gray-800 rounded-full">
                          +{user.permissions.length - 2} daha
                        </span>
                      )}
                    </div>
                  </td>
                  <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
                    {user.lastLoginAt 
                      ? user.lastLoginAt.toLocaleDateString('tr-TR')
                      : 'Hiç giriş yapmadı'
                    }
                  </td>
                  <td className="px-6 py-4 whitespace-nowrap">
                    <span className={`inline-flex px-2 py-1 text-xs font-semibold rounded-full ${
                      user.isActive 
                        ? 'bg-green-100 text-green-800' 
                        : 'bg-red-100 text-red-800'
                    }`}>
                      {user.isActive ? 'Aktif' : 'Pasif'}
                    </span>
                  </td>
                  <td className="px-6 py-4 whitespace-nowrap text-sm font-medium">
                    <div className="flex items-center space-x-2">
                      <button className="text-blue-600 hover:text-blue-900" title="Görüntüle">
                        <Eye className="h-4 w-4" />
                      </button>
                      <button className="text-gray-600 hover:text-gray-900" title="Düzenle">
                        <Edit className="h-4 w-4" />
                      </button>
                      <button className="text-purple-600 hover:text-purple-900" title="Yetkileri Düzenle">
                        <Settings className="h-4 w-4" />
                      </button>
                      <button className="text-red-600 hover:text-red-900" title="Sil">
                        <Trash2 className="h-4 w-4" />
                      </button>
                    </div>
                  </td>
                </tr>
              ))}
            </tbody>
          </table>
        </div>

        {filteredUsers.length === 0 && (
          <div className="text-center py-12">
            <User className="mx-auto h-12 w-12 text-gray-400" />
            <h3 className="mt-2 text-sm font-medium text-gray-900">Kullanıcı bulunamadı</h3>
            <p className="mt-1 text-sm text-gray-500">
              {searchTerm 
                ? 'Arama kriterlerinize uygun kullanıcı bulunamadı.'
                : 'Bu şirket için henüz kullanıcı eklenmemiş.'
              }
            </p>
            <button
              onClick={() => setShowAddUser(true)}
              className="mt-6 bg-purple-600 text-white px-4 py-2 rounded-lg hover:bg-purple-700 flex items-center mx-auto"
            >
              <UserPlus className="mr-2 h-4 w-4" />
              İlk Kullanıcıyı Ekle
            </button>
          </div>
        )}
      </div>

      {/* Add User Modal - Bu kısım daha sonra ayrı bir component olarak çıkarılabilir */}
      {showAddUser && (
        <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50">
          <div className="bg-white rounded-lg p-6 w-full max-w-md">
            <h3 className="text-lg font-semibold text-gray-900 mb-4">Yeni Kullanıcı Ekle</h3>
            <p className="text-gray-600 mb-4">
              {customer.companyName} için yeni kullanıcı ekleme formu burada olacak.
            </p>
            <div className="flex justify-end space-x-3">
              <button
                onClick={() => setShowAddUser(false)}
                className="px-4 py-2 border border-gray-300 text-gray-700 rounded-lg hover:bg-gray-50"
              >
                İptal
              </button>
              <button className="px-4 py-2 bg-purple-600 text-white rounded-lg hover:bg-purple-700">
                Kullanıcı Ekle
              </button>
            </div>
          </div>
        </div>
      )}
    </div>
  );
}